/* kXML
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License
 * on the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of kXML is Stefan Haustein. Copyright (C)
 * 2000, 2001 Stefan Haustein, D-46045 Oberhausen (Rhld.),
 * Germany. All Rights Reserved.
 *
 * Contributor(s): Paul Palaszewski, Wilhelm Fitzpatrick, 
 *                 Eric Foster-Johnson, Michael Angel
 *
 * */

package org.kxml.io;

import org.kxml.*;
import java.io.*;


/** A simple, pull based XML parser. This classes replaces the
    deprecated XmlParser class and the corresponding event classes. */

public class XmlReader extends AbstractXmlReader {

    static final String UNEXPECTED_EOF = "Unexpected EOF"; 
    
    char [] buf = new char [Runtime.getRuntime ().freeMemory () >= 1048576 
			   ? 8192 : 128];
    boolean eof;
    int bufPos;
    int bufCount;
    Reader reader;

    int line = 1;
    int column = 1;
    
    
    
    private int peekChar () throws IOException {
	if (eof) return -1;
	
	if (bufPos >= bufCount) {
	    bufCount = reader.read (buf, 0, buf.length);
	    bufPos = 0;
	    if (bufCount == -1) {
		eof = true;
		return -1;
	    }
	}
	
	return buf[bufPos];
    }       
    
    private int readChar () throws IOException {
	int p = peekChar ();
	bufPos++;
	column++;
	if (p == 10) {
	    line++;
	    column = 1;
	}
	return p;
    }
    
    
    private void skipWhitespace () throws IOException {
	while (!eof && peekChar () <= ' ')
	    readChar ();
    }
    
    
    
    private String readName () throws IOException {
	
	int c = readChar ();
	if (c < 128 && c != '_' && c != ':' 
	    && (c < 'a' || c > 'z')
	    && (c < 'A' || c > 'Z'))
	    exception ("name expected!");
	
	
	StringBuffer buf = new StringBuffer ();	
	buf.append ((char) c);
	
	while (!eof) {
	    c = peekChar ();
	    
	    if (c < 128 && c != '_' && c != '-' && c != ':' && c != '.' 
		&& (c < '0' || c > '9')
		&& (c < 'a' || c > 'z')
		&& (c < 'A' || c > 'Z'))

		break;
	    
	    buf.append ((char) readChar ());
	}
	
	return buf.toString ();
    }
    
    
    /** Reads chars to the given buffer until the given stopChar 
	is reached. The stopChar itself is not consumed. */
    
    private StringBuffer readTo (char stopChar,
			 StringBuffer buf) throws IOException {
	
	while (!eof && peekChar () != stopChar) 
	    buf.append ((char) readChar ());
	
	return buf;
    }
    
    

    
    public XmlReader (Reader reader) throws IOException {
	
	this.reader = reader; //new LookAheadReader (reader); 
    }
    
    
    
    /* precondition: &lt;!- consumed */
    
    private void parseComment () throws IOException {
	
	StringBuffer buf = new StringBuffer ();
	
	if (readChar () != '-') 
	    exception ("'-' expected");
	
	int cnt;
	int lst;
	
	while (true) {
	    readTo ('-', buf);
	    
	    if (readChar () == -1) 
		exception (UNEXPECTED_EOF);
	    
	    cnt = 0;

	    do {
		lst = readChar ();
		cnt++;  // adds one more, but first is not cnted
	    }
	    while (lst == '-');
	    
	    if (lst == '>' && cnt >= 2) break;
	    
	    while (cnt-- > 0) 
		buf.append ('-');
	    
	    buf.append ((char) lst);
	}
	
	while (cnt-- > 2) 
	    buf.append ('-');
	
	type = Xml.COMMENT;
	text = buf.toString ();
	name = null;
	namespace = null;
    }
    

    /* precondition: &lt! consumed */ 
    
    
    private void parseDoctype () throws IOException {
	
	StringBuffer buf = new StringBuffer ();
	int nesting = 1;
	
	while (true) {
	    int i = readChar ();
	    switch (i) {
	    case -1:
		exception (UNEXPECTED_EOF);
	    case '<': 
		nesting++;
		break;
	    case '>':
		if ((--nesting) == 0) {
		    namespace = null;
		    name = null;
		    type = Xml.DOCTYPE;
		    text = buf.toString ();
		    break;
		}
		buf.append ((char) i);
	    }
	}
    }
    
    
    private void parseCData () throws IOException {
	
	StringBuffer buf = readTo ('[', new StringBuffer ());
	
	if (!buf.toString ().equals ("CDATA")) 
	    exception ("Invalid CDATA start!");
	
	buf.setLength (0);

	readChar (); // skip '['
	
	int c0 = readChar ();
	int c1 = readChar ();
	
	while (true) {
	    int c2 = readChar ();
	    
	    if (c2 == -1) 
		exception (UNEXPECTED_EOF);
	    
	    if (c0 == ']' && c1 == ']' && c2 == '>') 
		break;
	    
	    buf.append ((char) c0);
	    c0 = c1;
	    c1 = c2;
	}
    
	type = Xml.TEXT;
	text = buf.toString ();
	name = null;
	namespace = null;
    }
    
    
    
    /* precondition: &lt;/ consumed */

    private void parseEndTag () throws IOException {
	
	skipWhitespace ();
	name = readName ();
	skipWhitespace ();
	
	if (readChar () != '>') 
	    exception ("'>' expected");
	
	type = Xml.END_TAG;
	text = null;
    }

    
    /** precondition: <? consumed */
    
    private void parsePI () throws IOException {
	StringBuffer buf = new StringBuffer ();
	readTo ('?', buf);
	readChar (); // ?
	
	while (peekChar () != '>') {
	    buf.append ('?');
	    
	    int r = readChar ();

	    if (r == -1) 
		exception (UNEXPECTED_EOF);
	    
	    buf.append ((char) r);
	    readTo ('?', buf);
	    readChar ();
	}
	
	readChar ();  // consume >
    
	type = Xml.PROCESSING_INSTRUCTION;
	text = buf.toString ();
	name = null;
	namespace = null;
    }
    
    
    private void parseStartTag () throws IOException {
	//current = new StartTag (current, reader, relaxed);
	
	//prefixMap = parent == null ? new PrefixMap () : parent.prefixMap;
	
	namespace = null;
	name = readName ();
	type = Xml.START_TAG;
	text = null;

	
	//System.out.println ("name: ("+name+")");
	
	//	degenerated = false;
	
	while (true) { 
	    skipWhitespace ();
	    
	    int c = peekChar ();
	    
	    if (c == '/') {
		degenerated = true;
		readChar ();
		skipWhitespace ();
		if (readChar () != '>') 
		    exception
			("illegal element termination");
		break;
	    }
	    
	    if (c == '>') { 
		readChar ();
		break;
	    }
	    
	    if (c == -1) 
		exception (UNEXPECTED_EOF); 
	    
	    String attrName = readName ();
	    
	    if (attrName.length() == 0)
		exception
                    ("illegal char / attr");
	    
	    skipWhitespace ();
	    
	    if (readChar () != '=') 
		exception
		    ("Attribute name "+attrName
		     +"must be followed by '='!");
	    
	    skipWhitespace ();
	    int delimiter = readChar ();
	    
	    if (delimiter != '\'' && delimiter != '"') {
		if (!relaxed)
		    exception
			("<" + name + ">: invalid delimiter: " 
			 + (char) delimiter);  
		
		
		delimiter = ' ';
	    }
	    
	    StringBuffer buf = new StringBuffer ();
	    readText (buf, (char) delimiter);
	    
	    if (attributeCount == attributes.length) {
		Attribute [] bigger = new Attribute [attributeCount + 8];
		System.arraycopy (attributes, 0, bigger, 0, attributeCount);
		attributes = bigger;
	    }
	    
	    attributes [attributeCount++] = 
		new Attribute (null, attrName, buf.toString ());
	
	    if (delimiter != ' ')
		readChar ();  // skip endquote
	}       
    }
    
    


    private int readText (StringBuffer buf, 
			  char delimiter) throws IOException {
	
	int type = Xml.WHITESPACE;
	int nextChar;
	
	
	while (true) {
	    nextChar = peekChar ();

	    if (nextChar == -1 
		|| nextChar == delimiter 
		|| (delimiter == ' ' 
		    && (nextChar == '>' || nextChar < ' '))) break;
	    
	    readChar ();
	    
	    if (nextChar == '&') {
		String code = readTo (';', new StringBuffer ()).toString ();
		readChar ();
		
		if (code.charAt (0) == '#') {
		    nextChar = (code.charAt (1) == 'x' 
				? Integer.parseInt (code.substring (2), 16) 
				: Integer.parseInt (code.substring (1)));
		    
		    if (nextChar > ' ') 
			type = Xml.TEXT;
		    
		    buf.append ((char) nextChar);           
		}
		else {
		    if (code.equals ("lt")) buf.append ('<');
		    else if (code.equals ("gt")) buf.append ('>');
		    else if (code.equals ("apos")) buf.append ('\'');
		    else if (code.equals ("quot")) buf.append ('"');
		    else if (code.equals ("amp")) buf.append ('&');
		    else {
			String resolved = characterEntityTable == null ?
			    null : (String) characterEntityTable.get (code);

			if (resolved == null) {
			    if (!relaxed) 
				exception ("Undef. &"+code+";");

			    resolved = "&"+code+";";
			}
			buf.append (resolved);
		    }
		    type = Xml.TEXT;
		}
	    }
	    else {
		if (nextChar > ' ') 
		    type = Xml.TEXT;
		buf.append ((char) nextChar);
	    }
	}
	
	return type;
    }    
    

    /** precondition: &lt; consumed */

    private void parseSpecial () throws IOException {
	
	switch  (peekChar ()) {
	case -1: exception (UNEXPECTED_EOF); 

	case '!':
	    readChar ();
	    switch (peekChar ()) {
	    case '-': 
		readChar ();
		parseComment ();
		break;

	    case '[':
		readChar ();
	        parseCData ();
		break;

	    default: 
		parseDoctype ();
		break;
	    }
	    break;

	case '?':
	    readChar ();
	    parsePI ();
	    break;

	case '/': 
	    readChar ();
	    parseEndTag ();
	    break;
	    
	default: 
	    parseStartTag ();
	}
    }



    public void nextImpl () throws IOException {
    
	attributeCount = 0;

	if (degenerated) {
	    type = Xml.END_TAG;
	    degenerated = false;
	}
	else {
	    switch (peekChar ()) {
		
	    case '<': 
		readChar ();
		parseSpecial ();
		break;
		
	    case -1: 
		type = Xml.END_DOCUMENT;
		name = null;
		namespace = null;
		text = null;
		break;
		
	    default: 
		{
		    StringBuffer buf = new StringBuffer ();
		    type = readText (buf, '<');
		    text = buf.toString ();
		    namespace = null;
		    name = null;
		}
	    }
	}
    }    

    
    
    /* default is false. Setting relaxed true
	allows CHTML parsing
    
    public void setRelaxed (boolean relaxed) {
	this.relaxed = relaxed;
    }
    */

    
    public int getLineNumber () {
	return line;
    }

    public int getColumnNumber () {
	return column;
    }

}

